/* LinearMemoryMap - write Theseus Linear Memory Map to timestamped file or stdout
   Avoids Theseus GUI window full issues
   Run as
     LinearMemoryMap
   to output to timestamped file (i.e. linear_memory_map-yyyymmdd-hhmm.txt)
   Run as
     linear_memory_map -
   to output to stdout
   Run as
     LinearMemoryMap file-name
   to output to named file
   Anything else will produce indeterminate results

   Copyright (c) 2012-2023 Steven Levine and Associates, Inc.
   All rights reserved.

   This program is free software licensed under the terms of the GNU
   General Public License, Version 2.  The GPL Software License can be found
   in gnugpl2.txt or at http://www.gnu.org/licenses/licenses.html#GPL

   2012-09-24 SHL Baseline
   2019-05-27 SHL Support -g and -v view options
   2020-11-19 SHL Correct comment typo
   2023-05-18 SHL Die to stderr to not hide error messages - use DieE
   2023-06-16 SHL Report errors to stderr to not hide error messages - use ErrorE
   2023-12-24 SHL Sync with templates
   2023-12-24 SHL Pass2 - ensure dir always defined before use
*/

signal on Error
signal on Failure name Error
signal on Halt
signal on NotReady name Error
signal on NoValue name Error
signal on Syntax name Error

gVersion = '0.1 2023-12-24'

Globals = 'gCmdName gEditor gEnv gOutputFile gVersion gViewGUI gViewVIO'

call Initialize

Main:
  parse arg cmdLine
  cmdLine = strip(cmdLine)

  select
  when cmdLine == '-' then
    call Pass2
  otherwise
    call Pass1 cmdLine
  end

  exit

/* end main */

/*=== Pass1() Redirect to stdout ===*/

Pass1: procedure expose (Globals)

  parse arg cmdTail
  cmdTail = strip(cmdTail)

  gOutputFile = ''
  gViewGUI = 0
  gViewVIO = 0

  do while cmdTail \= ''
    parse var cmdTail curArg cmdTail
    cmdTail = strip(cmdTail)
    select
    when curArg == '-g' then
      gViewGUI = 1
    when curArg == '-h' | curArg == '-?' then
      call ScanArgsHelp
    when curArg == '-v' then
      gViewVIO = 1
    when curArg == '-V' then do
      say gCmdName gVersion
      exit
    end
    when left(curArg, 1) = '-' then
      call ScanArgsUsage curArg 'unexpected'
    otherwise
      if gOutputFile \== '' then
	call ScanArgsUsage 'Output file already set to' gOutputFile
      gOutputFile = curArg
    end
  end /* do */

  /* Need full path because we change to Theseus directory */
  if gOutputFile == '' | gOutputFile == '.' then
    gOutputFile = MakePath(directory(), MakeTimestampedFileName(gCmdName || '.lst'))
  else if IsDir(gOutputFile) then
    gOutputFile = MakePath(gOutputFile, MakeTimestampedFileName(gCmdName || '.lst'))
  else if left(gOutputFile, 1) == '\' then
    gOutputFile = MakePath(left(directory(), 3), gOutputFile)	/* Supply drive letter */
  else if substr(gOutputFile, 2, 2) \= ':\' then
    gOutputFile = MakePath(directory(), gOutputFile)	/* Make relative path absolute */

  say 'Writing results to' gOutputFile

  parse source . . cmd
  if Is4OS2() then
    cmd = '@'cmd '-' '>'gOutputFile
  else do
    shell = value('COMSPEC',, gEnv);
    cmd = shell '/c' cmd '-' '>'gOutputFile
  end
  cmd

  if gViewVIO then do
    call FindEditor
    gEditor gOutputFile
  end

  if gViewGUI then do
    call FindGUIEditor
    /* RC = 1 expected */
    signal off Error
    gEditor gOutputFile
    signal on Error
  end

  return

/* end Pass1 */

/*=== Pass2() Write output to stdout ===*/

Pass2: procedure expose (Globals)

  /* Theseus directory must be in PATH and LIBPATH
   * or must run from Theseus directory
   */
  do 1
    exeName = 'theseus4.exe'
    queryResult = ''
    exePath = SysSearchPath('PATH', exeName)
    if exePath \== '' then leave	/* Assume in LIBPATH too */
    /* Check well known places */
    exePath = 'd:\devtools\theseus4\' || exeName
    queryResult = stream(exePath, 'C', 'QUERY EXISTS')
    if queryResult \== '' then leave
    exePath = 'd:\apps\theseus4\' || exeName
    queryResult = stream(exePath, 'C', 'QUERY EXISTS')
    if queryResult \== '' then leave
    call Die exeName 'not found in PATH or well-known places'
  end

  if queryResult == '' then
    dir = ''
  else do
    /* Need directory change */
    i = lastpos('\', queryResult)
    dir = substr(queryResult, 1, i - 1)	/* Chop \ and exeName from path */
    olddir = directory()
    call directory dir			/* Run from Theseus directory */
  end

  call WarnMsg 'Running from' directory()

  call RxFuncQuery 'RT2LoadFuncs1'
  if RESULT then do
    call RxFuncAdd 'RT2LoadFuncs1', 'THESEUS1', 'RT2LoadFuncs1'
    if RESULT then
      call Die 'RxFuncAdd failed for RT2LoadFuncs1 with RESULT' RESULT
  end
  /* FIXME to doc why RT2LoadFuncs1 must always be run - must do some extra init */
  call RT2LoadFuncs1
  if RESULT then
    call Die 'RT2LoadFuncs1 failed with error' RESULT

  nop

  /* Outputs to stdout */
  call RT2GetLinMemMap

  if dir \== '' then
    call directory olddir

  call WarnMsg 'Done'

  return

/* end Pass2 */

/*=== Initialize() Initialize globals ===*/

Initialize: procedure expose (Globals)
  call SetCmdName
  call LoadRexxUtil
  gEnv = 'OS2ENVIRONMENT'
  return

/* end Initialize */

/*=== ScanArgsHelp() Display ScanArgs usage help exit routine ===*/

ScanArgsHelp:
  say
  say 'Output Theseus Linear Memory Map to a file.'
  say 'Optionally view results in a GUI or VIO editor.'
  say
  say 'Usage:' gCmdName '[-g] [-h] [-v] [-V] [-?] output-file'
  say
  say '  -g           View results with GUI editor'
  say '  -h -?        Display this message'
  say '  -v           View results with VIO editor'
  say '  -V           Display version number and quit'
  say
  say '  output-file  Linear memory map output'
  exit 255

/* end ScanArgsHelp */

/*=== ScanArgsUsage(message) Report ScanArgs usage error exit routine ===*/

ScanArgsUsage:
  parse arg msg
  say
  if msg \== '' then
    say msg
  say 'Usage:' gCmdName '[-g] [-h] [-v] [-V] [-?] output-file'
  exit 255

/* end ScanArgsUsage */

/*==============================================================================*/
/*=== SkelRexxFunc standards - Delete unused - Move modified above this mark ===*/
/*==============================================================================*/

/*=== FindEditor() Find non-GUI editor and set gEditor or Die ===*/

FindEditor: procedure expose (Globals)
  /* Requires GetEnv */
  /* Requires IsExeInPath */
  /* Uses gEditor from Globals */
  /* Uses EDITOR from environment */
  do while symbol('gEditor') \== 'VAR'
    gEditor = GetEnv('EDITOR')
    if gEditor \== '' then leave
    gEditor = 'vim.exe'			/* Force ext in case have vim alias */
    if IsExeInPath(gEditor) then leave
    gEditor = '4os2 /c vimx'
    if IsExeInPath('vimx.cmd') then leave
    gEditor = 'tedit'
    if IsExeInPath(gEditor) then leave
    call Die 'EDITOR not defined and cannot guess editor to use'
  end /* while */
  return

/* end FindEditor */

/*=== FindGUIEditor() Find GUI editor ===*/

FindGUIEditor: procedure expose (Globals)
  /* Requires GetEnv */
  /* Requires IsExeInPath */
  /* Uses gEditor from Globals */
  /* Uses EDITOR from environment */
  /* Uses CPE from environment */
  do while symbol('gEditor') \== 'VAR'
    gEditor = GetEnv('EDITOR')
    if gEditor \== '' then leave
    gEditor = GetEnv('CPE')
    if gEditor \== '' then do
      gEditor = gEditor || '\bin\cpe'
      leave
    end
    gEditor = 'vim'
    if IsExeInPath(gEditor) then leave
    gEditor = 'vimx.cmd'
    if IsExeInPath(gEditor) then do
      gEditor = '4os2 /c vimx'
      leave
    end
    gEditor = 'tedit'
    if IsExeInPath(gEditor) then leave
    call Die 'EDITOR not defined and cannot guess usable GUI editor'
  end
  return

/* end FindGUIEditor */

/*=== GetEnv(var) Return value for environment variable or empty string ===*/

GetEnv: procedure expose (Globals)
  parse arg var
  if var = '' then
    call Die 'GetEnv requires an argument'
  return value(var,, gEnv)

/* end GetEnv */

/*=== Is4OS2() Return true if running under 4OS2 else false ===*/

Is4OS2: procedure expose (Globals)
  /* Keep Is4OS2 and Chk4OS2 in sync */
  old = value('_X',, gEnv)		/* In case in use */
  /* 4OS2 sets _X to 0, cmd.exe sets x to @eval[0], rxd leaves X unchanged */
  '@set _X=%@eval[0]'
  new = value('_X',, gEnv)
  '@set _X=' || old			/* Restore */
  yes = new = 0 | old == new		/* Assume 4OS2 if running under rxd */
  return yes				/* if running under 4OS2 */

/* end Is4OS2 */

/*=== IsDir(dirName[, full]) return true if directory is valid, accessible directory ===*/

IsDir: procedure
  /* If arg(2) not omitted, return full directory name or empty string */
  parse arg dir, full
  newdir = ''

  do 1
    if dir == '' then do
      cwd = ''				/* No restore needed */
      leave
    end
    dir = translate(dir, '\', '/')	/* Convert to OS/2 slashes */
    s = strip(dir, 'T', '\')		/* Chop trailing slashes unless root */
    if s \== '' & right(s, 1) \== ":" then
      dir = s				/* Chop */
    drv = filespec('D', dir)
    cwd = directory()			/* Remember */
    /* If have drive letter and requested directory on some other drive */
    if drv \== '' & translate(drv) \== translate(left(cwd, 2)) then do
      /* Avoid slow failures and unwanted directory changes */
      drvs = SysDriveMap('A:')
      if pos(translate(drv), drvs) = 0 then
	leave				/* Unknown drive */
      if SysDriveInfo(drv) == '' then
	leave				/* Drive not ready */
      cwd2 = directory(drv)		/* Remember current directory on other drive */
      newdir = directory(dir)		/* Try to change and get full path name */
      call directory cwd2		/* Restore current directory on other drive */
      leave
    end

    /* If no drive letter or same drive and not UNC name */
    if left(dir, 2) \== '\\' then do
      newdir = directory(dir)		/* Try to change and get full path name */
      leave
    end

    /* UNC name - hopefully server is accessible or this will be slow
       Accept
	 \\server
	 \\server\
	 \\server\dir\
	 \\server\dir
     */
    cwd = ''				/* No restore needed */
    wc = dir
    if right(wc, 1) \== '\' then
      wc = wc || '\'
    i = lastpos('\', wc)
    if substr(wc, 3, 1) == '\' then
      leave				/* Malformed UNC - no server name */
    if pos('*', wc) > 0 | pos('?', wc) > 0 then
      leave				/* No wildcards allowed */
    call SysFileTree wc, 'files', 'O'
    if files.0 > 0 then do
      s = files.1			/* Exists and is not empty */
      i = lastpos('\', s)
      newdir = left(s, i - 1)		/* Extract directory name from full path name */
      leave
    end
    /* Try wildcarded directory name */
    wc = strip(wc, 'T', '\')
    i = lastpos('\', wc)
    base = substr(wc, i + 1)
    if base == '' then
      leave				/* Should have matched above */
    wc = substr(wc, 1, i) || '*' || base || '*'
    call SysFileTree wc, 'files', 'DO'
    do fileNum = 1 to files.0
      /* Find directory name is list */
      s = files.fileNum
      i = lastpos('\', s)
      s2 = substr(s, i + 1)
      if translate(base) == translate(s2) then do
	newdir = left(s, i - 1)
	leave
      end
    end /* i */
  end /* 1 */

  if cwd \== '' then
    call directory cwd			/* Restore original directory and drive */

  if full \== '' then
    ret = newdir			/* Return full directory name or empty string */
  else
    ret = newdir \== ''			/* Return true if valid and accessible */
  return ret

/* end IsDir */

/*=== IsExeInPath(exe, wantPath) return TRUE if executable is in PATH or full path ===*/

/**
 * @param exe is executable to find, .exe assumed if no extension specified
 * @param wantPath optionally requests full path return
 * @returns TRUE if executable found in PATH, FALSE if not
 * @returns full path or empty string if wantPath not empty
 */

IsExeInPath: procedure
  parse arg exe, wantPath
  if exe \ == '' then do
    i = lastpos('.', exe)
    j = lastpos('\', exe)
    if i = 0 | i < j then
      exe = exe || '.exe'		/* No extension */
    exe = SysSearchPath('PATH', exe)
  end
  if wantPath == '' then
    exe = exe \== ''			/* Want bool result */

  return exe				/* Return bool or full path */

/* end IsExeInPath */

/*=== MakePath(pathparts,...) Make path name from parts ===*/

MakePath: procedure

  /* All parts optional - code guesses what caller means.
     If last arg begins with a dot and is not .. and does not
     contain a slash, it is assumed to be a file extension.
     To avoid this behavior, pass empty arg as last arg.
     Empty args are ignored.
     Automatically converts unix slashes to dos slashes.
     If 1st arg is drive letter, it must have trailing colon.
   */

  argCnt = arg()

  path = ''

  do argNum = 1 to argCnt
    s = arg(argNum)
    s = translate(s, '\', '/')		/* Ensure DOS */
    if s == '' & argNum = argCnt then
      iterate				/* Ignore nul last arg */
    if argNum = 1 then
      path = s
    else do
      lead = left(s, 1)
      tail = right(path, 1)
      if tail == ':' & argNum = 2 then
	path = path || s		/* Append path part to drive spec */
      else if lead == '.' & argNum = argCnt & s \== '..' & pos('\', s) = 0  then
	path = path || s		/* Assume extension unless .. or contains \ */
      else if tail == '\' & lead == '\' then
	path = path || substr(s, 2)	/* Drop extra backslash */
      else if path \== '' & tail \== '\' & lead \== '\' then
	path = path || '\' || s		/* Ensure have backslash */
      else
	path = path || s
    end
  end /* for */

  return path

/* end MakePath */

/*=== MakeTimestampedFileName(fileName, short) Return timestamped file name ===*/

MakeTimestampedFileName: procedure expose (Globals)

  parse arg fileName, short

  short = short \== '' & short \= 0

  /* Insert yyyymmdd-hhmm or yyyymmdd timestamp between file name and extension */

  if fileName = '' then
    call Die 'MakeTimestampedFileName requires a file name argument'

  /* Generate yyyymmdd-hhmm */
  if short then
    s = date('S')
  else
    s = date('S') || '-' || left(space(translate(time(),,':'),0), 4)

  i = lastpos('.', fileName)
  j = lastpos('\', fileName)
  /* Generate name-yyyymmdd-hhmm.ext */
  if i = 0 | i < j then
    s = fileName || '-' || s		/* No extension */
  else
    s = substr(fileName, 1, i - 1) || '-' || s || substr(fileName, i)

  return s

/* end MakeTimestampedFileName */

/*=== WarnMsg(message,...) Write multi-line warning message to STDERR ===*/

WarnMsg: procedure
  do i = 1 to arg()
    msg = arg(i)
    call lineout 'STDERR', msg
  end
  return

/* end WarnMsg */

/*==========================================================================*/
/*=== SkelRexx standards - Delete unused - Move modified above this mark ===*/
/*==========================================================================*/

/*=== ErrorE() Set gErrCondition; report to STDERR; trace and exit or return if called ===*/

Error:
  say
  parse source . . cmd
  gErrCondition = condition('C')
  call lineout 'STDERR', gErrCondition 'signaled at' cmd 'line' SIGL || '.'
  if condition('D') \== '' then
    call lineout 'STDERR', 'REXX reason =' condition('D') || '.'
  if gErrCondition == 'SYNTAX' & symbol('RC') == 'VAR' then
    call lineout 'STDERR', 'REXX error =' RC '-' errortext(RC) || '.'
  else if symbol('RC') == 'VAR' then
    call lineout 'STDERR', 'RC =' RC || '.'
  call lineout 'STDERR', 'Source ="' || strip(sourceline(SIGL)) || '".'

  if condition('I') \== 'CALL' | gErrCondition == 'NOVALUE' | gErrCondition == 'SYNTAX' then do
    trace '?A'
    call lineout 'STDERR', 'Exiting.'
    nop
    if symbol('RC') \== 'VAR' then
      RC = 255
    exit RC
  end
  return

/* end ErrorE */

/*=== DieE([message,...]) Write multi-line message to STDERR and die ===*/

Die:
  call lineout 'STDERR', ''
  do i = 1 to arg()
    call lineout 'STDERR', arg(i)
  end
  call lineout 'STDERR', gCmdName 'aborting.'
  call beep 200, 300
  call SysSleep 2
  exit 254

/* end DieE */

/*=== SetCmdName() Set gCmdName to short script name ===*/

SetCmdName: procedure expose (Globals)
  parse source . . cmd
  cmd = filespec('N', cmd)		/* Chop path */
  c = lastpos('.', cmd)
  if c > 1 then
    cmd = left(cmd, c - 1)		/* Chop extension */
  gCmdName = translate(cmd, xrange('a', 'z'), xrange('A', 'Z'))	/* Lowercase */
  return

/* end SetCmdName */

/*=== Halt() Report HALT condition to STDOUT and exit ===*/

Halt:
  say
  parse source . . cmd
  say condition('C') 'signaled at' cmd 'line' SIGL || '.'
  say 'Source =' sourceline(SIGL)
  say 'Sleeping for 2 seconds...'
  call SysSleep 2
  exit 253

/* end Halt */

/*=== LoadRexxUtil() Load RexxUtil functions ===*/

LoadRexxUtil:
  if RxFuncQuery('SysLoadFuncs') then do
    call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
    if RESULT then
      call Die 'Cannot load SysLoadFuncs.'
    call SysLoadFuncs
  end
  return

/* end LoadRexxUtil */

/* The end */
